/////////////////////////////////////////////////////////////////////////////
/*
    no-ip.com dynamic IP update client for Linux

   Copyright (C) 2000 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


			written June 2000
			by johna@onevista.com

	copyright transferred from 
		One Vista Associates 
	to 
		Free Software Foundation, Inc. 
	October 6, 2000 (johna)

	+	November 4, 2000
	+	updated noip.c and Makefile to build on solaris also
	+	collapsed expanded code into strerror()
	+	suggested by rob_nielson@mailandnews.com
	
	+	December 2, 2000
	+	updated noip.c to build on BSD also
	+	changed #include <linux/if.h> to #include <net/if.h>
	+	suggested by ramanan@users.sourceforge.net

	+	April 27, 2001 (Stephane Neveu stephane@skylord.org)
	+	changed the "SAVEDIPFILE" from /tmp/no-ip_save to 
		/var/run/no-ip_save
	+	added configuration default lookup into /usr/local/etc
		if /usr/local/lib doesn't have a configuration file
	+	fix output of contextual ourname[hostnum] in the function
		handle_dynup_error() instead of the "first" name

	+	August 27, 2001 (johna)
	+	added GROUP concept
	+	removed multiple host/domain name hack (use groups)
	+	changed SAVEDIPFILE back to /tmp 
			(must be root to write in /var/run)

*/			
/////////////////////////////////////////////////////////////////////////////                                            

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <syslog.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <net/if.h>

#ifdef linux
 #ifndef SIOCGIFADDR
  #include <linux/sockios.h>
 #endif
#endif

#ifdef sun
 #include <sys/sockio.h>
#endif

#define DEBUG

#define READ_TIMEOUT		90
#define WRITE_TIMEOUT		60
#define CONNECT_TIMEOUT		60
#define STD_PORT		80
#define PROXY_PORT		8245
#define CLIENT_IP_PORT		8245
#define OURBUFLEN		8192
#define IPLEN			16
#define SAVEDIPFILE		"/tmp/no-ip_save"
#define CONFIG_FILENAME		"/usr/local/lib/no-ip.conf"
#define CONFIG_FILENAME_ETC     "/usr/local/etc/no-ip.conf"
#define CONFIGNUM		10
#define CONFSTRLEN		1024
#define NUMBER			2
#define STRING			4
#define BOOL			8
#define SPACE			' '
#define EQUALS			'='
#define COMMENT			'#'
#define COMMA                   ','

#define ALREADYSET               0
#define SUCCESS                  1
#define BADHOST                  2
#define BADPASSWD                3
#define BADUSER                  4
#define TOSVIOLATE               6
#define PRIVATEIP                7
#define HOSTDISABLED             8
#define HOSTISREDIRECT           9
#define BADGRP                  10
#define SUCCESSGRP              11
#define ALREADYSETGRP           12
#define RELEASEDISABLED         99

#define UNKNOWNERR		-1
#define NOHOSTLOOKUP		-2
#define SOCKETFAIL		-3
#define CONNTIMEOUT		-4
#define CONNFAIL		-5
#define READTIMEOUT		-6
#define READFAIL		-7
#define WRITETIMEOUT		-8
#define WRITEFAIL		-9
#define NOCONFIG		-10
#define BADCONFIG1		-11
#define BADCONFIG2		-12
#define BADCONFIG3		-13
#define BADCONFIG4		-14
#define BADCONFIG5		-15
#define BADCONFIG6		-16
#define BADCONFIG8		-18
#define BADCONFIG9		-19

#define MAX(x,y)		(((x)>(y))?(x):(y))

int	debug			= 	0;
int	timed_out		=	0;
int	conf_line		=	0;
int	port_to_use		=	STD_PORT;
int	socket_fd		=	-1;
int	background		=	0;
int	proxy			=	0;
int	nat			=	0;
int	interval		=	30;
int	log2syslog		= 	1;
int	connect_fail		=	0;
char	*noipname		=	"dynupdate.no-ip.com";
char	*program		=	"noip";
char	*version		=	"1.6.0";
char	*os			=	"Linux";
char	error_filename[20]	=	"/tmp/NO-IPXXXXXX";
char	*config_filename	=	NULL;
char	IPaddress[IPLEN];
char	suppliedIP[IPLEN+6];
char	login[CONFSTRLEN];
char	password[CONFSTRLEN];
char	hostname[CONFSTRLEN];
char	domain[CONFSTRLEN];
char	groupname[CONFSTRLEN];
char	device[CONFSTRLEN];
char	ourname[128];
char	Last_IP_Addr[IPLEN];
char	our_visible_IPaddr[IPLEN];
char	buffer[OURBUFLEN];
struct	sigaction sig;

struct CONFIG {
	char	*keyword;
	void	*dest;			// typeless pointer !!
	int	type;
	int	seen;
} config[CONFIGNUM] = {
	{"LOGIN",	login,    	STRING  },
	{"PASSWORD",	password, 	STRING  },
	{"HOSTNAME",	hostname,       STRING	},
	{"DOMAIN",	domain,   	STRING	},
	{"GROUP",	groupname,   	STRING	},
	{"PROXY",	&proxy,	  	BOOL    },
	{"NAT",		&nat,	  	BOOL    },
	{"INTERVAL",	&interval,      NUMBER  },
	{"DAEMON",	&background,   	BOOL    },
	{"DEVICE",	device,   	STRING  }
};

///////////////////////////////////////////////////////////////////////////
void	process_options(int argc, char *argv[]);
void	alarm_handler(int sig);
void	save_IP(int sig);
void	get_our_visible_IPaddr(char *dest);
void	getip(char *p);
int	run_as_background();
int	Read(int sock, char *buf, size_t count);
int	Write(int fd, char *buf, size_t count);
int	Connect(int port);
int	converse_with_web_server();
char	*despace(char *p);
int	parse_config();
int	dynamic_update();
int	handle_dynup_error(int error_code);
int	handle_config_error(int err_code);
void	ErrMsg(char *fmt, ...);
///////////////////////////////////////////////////////////////////////////
void Usage()
{
	fprintf(stderr, "USAGE: %s [-c config_file] [-h] [-l] [-i IPaddress]",
							program);
#ifdef DEBUG
	fprintf(stderr, " [-d]");
#endif
	fprintf(stderr, "\n\nVersion %s-%s\n",os, version);
	fprintf(stderr, "Options: -c config_file   use alternate config file\n");
	fprintf(stderr, "         -h               help\n");
	fprintf(stderr, "         -i IPaddress     use supplied address\n");
	fprintf(stderr, "         -l               log output to stderr\n");
#ifdef DEBUG
	fprintf(stderr, "         -d               increase debug verbosity\n");
#endif
	fprintf(stderr, "\n   config file is either\n");
	fprintf(stderr, "      %s or %s\n", CONFIG_FILENAME,CONFIG_FILENAME_ETC);
	fprintf(stderr, "      and must contain %d required entries", CONFIGNUM);
	fprintf(stderr, " in the form of KEYWORD = value\n");
	fprintf(stderr, "      A complete description of the config file is in the README file.\n");
	fprintf(stderr, "\tKEYWORD    value \t\t\t (example)\n\n");
	fprintf(stderr, "\tLOGIN    = YourNo-ip.comLogin \t\t (foo@host.org)\n");
	fprintf(stderr, "\tPASSWORD = YourNo-ip.comPassword \t (hello25-1)\n");
	fprintf(stderr, "\tHOSTNAME = YourNo-ip.comHost	\t (grumpy)\n");
	fprintf(stderr, "\tDOMAIN   = YourNo-ip.comDomain \t\t (no-ip.com)\n");
	fprintf(stderr, "\tGROUP    = YourNo-ip.comGroup \t\t (mygoup)\n");
	fprintf(stderr, "\tDEVICE   = YourInternetConnectionDevice  (ppp0)\n");
	fprintf(stderr, "\tPROXY    = Y/N \t\t\t\t (Y)\n");
	fprintf(stderr, "\tNAT      = Y/N \t\t\t\t (N) \n");
	fprintf(stderr, "\tDAEMON   = Y/N \t\t\t\t (Y)\n");
	fprintf(stderr, "\tINTERVAL = Minutes\t\t\t (30)\n");
}
///////////////////////////////////////////////////////////////////////////
int main(int argc, char *argv[])
{
        int i;
	char *p;

	p = strrchr(argv[0], '/');
	if (p)
	    program = ++p;
	else
	    program = argv[0];
	openlog(program, LOG_PID, LOG_DAEMON);
	process_options(argc, argv);
	if (config_filename == NULL) {
		if(!access(CONFIG_FILENAME, R_OK)) 
		 	config_filename = CONFIG_FILENAME;
		 else
			config_filename = CONFIG_FILENAME_ETC;
	}
	if (handle_config_error(parse_config()) != SUCCESS)
	    return -1;
	if (*groupname)
	    sprintf(ourname, "%s", groupname);
	else {
	    if (*hostname) {
		if (*domain)
		    sprintf(ourname, "%s.%s", hostname, domain);
		else
		    return handle_config_error(BADCONFIG8);
	    } else
		return handle_config_error(BADCONFIG9);
	}
	if (*suppliedIP != 0) {
	    if (background) {
		ErrMsg("IP address detected on command line while configured for daemon mode.");
		ErrMsg("Running in single use mode.");
		background = 0;
	    }
	}
	if (proxy)
	    port_to_use = PROXY_PORT;
	timed_out = 0;
	sig.sa_flags = 0;
	sigemptyset(&sig.sa_mask);
	sig.sa_handler = SIG_IGN;
	sigaction(SIGHUP,&sig,NULL);
	sigaction(SIGPIPE,&sig,NULL);
	sigaction(SIGUSR1,&sig,NULL);
	sigaction(SIGUSR2,&sig,NULL);
	sig.sa_handler = alarm_handler;
	sigaction(SIGALRM,&sig,NULL);
	if (background) {
	    sig.sa_handler = save_IP;
	    sigaction(SIGINT,&sig,NULL);
	    sigaction(SIGTERM,&sig,NULL);
	    sigaction(SIGQUIT,&sig,NULL);
	    run_as_background();
	} else {
	    if (*IPaddress == 0) {
		if (nat)
		    get_our_visible_IPaddr(IPaddress);
		else
		    getip(IPaddress);
	    }
	    if (*IPaddress == 0) {
		ErrMsg("Not connected to Net!");
		return 1;
	    }
	     if (handle_dynup_error(dynamic_update()) != SUCCESS)
		return -1;
	    save_IP(0);
	}
	return 0;
}
///////////////////////////////////////////////////////////////////////////
void process_options(int argc, char *argv[])
{
	extern  int     optind, opterr;
	extern  char    *optarg;
	int     c;

	while ((c = getopt(argc, argv, "c:dhi:l")) != EOF)	{
		switch (c) {
		case 'c':
			config_filename = optarg;
			break;
		case 'd':
#ifdef DEBUG
			debug++;
			log2syslog = 0;
#endif
			break;
		case 'h':
			Usage();
			exit(0);
		case 'i':
			strcpy(IPaddress, optarg);
			sprintf(suppliedIP, "&ip=%s", optarg);
			break;
		case 'l':
			log2syslog = 0;
			break;
		default:
			Usage();
			exit(0);
		}
	}
	if (argc - optind) {
	    Usage();
	    exit(1);
	}
	return;
}
///////////////////////////////////////////////////////////////////////////
void alarm_handler(int sig)
{
	timed_out = 1;
}
///////////////////////////////////////////////////////////////////////////
void save_IP(int sig)
{			// save Last_IP_Addr on SIGTERM & SIGQUIT
	int fd;
	
	fd = open(SAVEDIPFILE, O_RDWR|O_CREAT|O_TRUNC, 0644);
	if (fd >= 0) {
	    write(fd, Last_IP_Addr, strlen(Last_IP_Addr) + 1);
	    close(fd);
	}
	exit(0);
}
///////////////////////////////////////////////////////////////////////////
void getip(char *p)
{
	int	fd;
	int	*x;
	struct	ifreq ifr;
	struct	in_addr z;

	*p = 0;		// remove old address
	if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
	    ErrMsg("Can't talk to kernel! (%d)\n", errno);
	    return;
	}
	strcpy(ifr.ifr_name, device);
	if (ioctl(fd, SIOCGIFFLAGS, &ifr) < 0)  {
	    ErrMsg("Can't get status for %s. (%d)\n", device, errno);
	    close(fd);
	    return;
	}
	if ((ifr.ifr_flags & IFF_UP) == 0)  {
// No longer print message when interface down  (johna 6-28-00)
//	    ErrMsg("Interface %s not active.\n", device);
	    close(fd);
	    return;
	}
	if (ioctl(fd, SIOCGIFADDR, &ifr) != 0) {
	    ErrMsg("Can't get IP address for %s. (%d)\n", device, errno);
	    close(fd);
	    return;
	}
	close(fd);
	x = (int *)&ifr.ifr_addr;	// seems to be off by 2 bytes
	z.s_addr = x[1];
	strcpy(p, inet_ntoa(z));
	sprintf(suppliedIP, "&ip=%s", p);  // force sending local address
#ifdef DEBUG
	if (debug)
	    fprintf(stderr,"! Our IP address is %s\n", p);
#endif
}
/////////////////////////////////////////////////////////////////////////
int run_as_background()
{
	int	i, x, delay;
	char	*err_string, *fp;
	FILE	*fildes;

	x = fork();
	switch (x) {
	    case -1:		// error
		err_string = strerror(errno);
		ErrMsg( "Can't fork!! (%s) Ending!\n", err_string);
		return -1;
	    default:		// parent
		exit(0);
	    case 0:		//child
		setsid();
		syslog(LOG_INFO, "%s started as a background process%s\n", 
			program, (nat) ? ((proxy) ? 
			" with NAT and PROXY enabled" : " with NAT enabled")
			: "");
		fp = NULL;
		fildes = fopen(SAVEDIPFILE, "r");
		if (fildes != NULL) {
		    fp = fgets(buffer, OURBUFLEN, fildes);
		    fclose(fildes);
		}
		if (fp != NULL)
		    strncpy(Last_IP_Addr, buffer, IPLEN);
		else
		    memset(Last_IP_Addr, 1, IPLEN);  // bogus address at start	
		unlink(SAVEDIPFILE);
		while (background) {
		    delay = MAX(60, interval * 60);
	            if (connect_fail)
        		delay += 500;
		    if (nat)
			get_our_visible_IPaddr(IPaddress);
		    else
			getip(IPaddress);
#ifdef DEBUG
		    ErrMsg("! LIA = %s, IP = %s",Last_IP_Addr, IPaddress);
#endif
		    if ((*IPaddress != 0) && 
				(strcmp(IPaddress, Last_IP_Addr) != 0)) {
			x = handle_dynup_error(dynamic_update());
		        if (x == SUCCESS)	// keep last successful IP
			    strcpy(Last_IP_Addr, IPaddress);
			else
			    break;
		    }
		    sleep(delay);
		}
		break;
	}
	return x;
}
/////////////////////////////////////////////////////////////////////////
int Read(int sock, char *buf, size_t count)
{
	size_t bytes_read = 0;
	int i;
	
	timed_out = 0;
	while (bytes_read < count) {
		alarm(READ_TIMEOUT);
		i = read(sock, buf, (count - bytes_read));
		alarm(0);
		if (timed_out) { 
		    if (bytes_read) {
			syslog(LOG_WARNING,"Short read from %s", noipname);
			return bytes_read;
		    } else
			return READTIMEOUT;
		}
		if (i < 0)
			return READFAIL;
		if (i == 0)
			return bytes_read;
		bytes_read += i;
		buf += i;
	}
	return count;
}
///////////////////////////////////////////////////////////////////////////
int Write(int fd, char *buf, size_t count)
{
	size_t bytes_sent = 0;
	int i;

	timed_out = 0;
	while (bytes_sent < count) {
		alarm(WRITE_TIMEOUT);
		i = write(fd, buf, count - bytes_sent);
		alarm(0);
		if (timed_out)
			return WRITETIMEOUT;
		if (i <= 0) 
			return WRITEFAIL;
		bytes_sent += i;
		buf += i;
	}
	return count;
}
///////////////////////////////////////////////////////////////////////////
int Connect(int port)
{
	int	fd, i;
	struct	in_addr saddr;
	struct	sockaddr_in addr;
	struct	hostent *host;

	host = gethostbyname(noipname);
	if (!host)
		return NOHOSTLOOKUP;
	memcpy(&saddr.s_addr, host->h_addr_list[0], 4);
	memset((char *) &addr, 0, sizeof(addr));
	addr.sin_family = AF_INET;
	addr.sin_port = htons(port);
	addr.sin_addr.s_addr = saddr.s_addr;
	fd = socket(AF_INET, SOCK_STREAM, 0);
	if (fd < 0)
		return SOCKETFAIL;
	timed_out = 0;
	alarm(CONNECT_TIMEOUT);
	i = connect(fd, (struct sockaddr *)&addr, sizeof(addr));
	alarm(0);
	if (i < 0)  {
	    if (timed_out)
		i = CONNTIMEOUT;
	    else
		i = CONNFAIL;
	    close(fd);		// remove old socket
	    connect_fail = 1;
	    return i;
	}
	socket_fd = fd;
	connect_fail = 0;
	return SUCCESS;
}
/////////////////////////////////////////////////////////////////////////
int converse_with_web_server()
{
	int	x;
#ifdef DEBUG
	char	*p, *q;

	if (debug) {
	    p = buffer;				// point at the first line
	    while ((q = strchr(p, '\n'))) {
		*q = 0;
		fprintf(stderr, "> %s\n", p);	// print the line
		*q++ = '\n';
		p = q;				// point at next line
	    }
	    if (*p)
		fprintf(stderr, "> %s\n", p);	// display the last line
	}
#endif
	if ((x = Write(socket_fd, buffer, strlen(buffer))) <= 0) {
	    close(socket_fd);
	    return x;
	}
	if ((x = Read(socket_fd, buffer, OURBUFLEN - 2)) < 0) {
	    close(socket_fd);
	    return x;
	}
	buffer[x++] = 0;		// terminate the response

#ifdef DEBUG
	if (debug) {
	    p = buffer;                 	// point at the first line
	    while ((q = strchr(p, '\n'))) {
		*q = 0;
		fprintf(stderr, "< %s\n", p);	// print the line
		*q++ = '\n';
		p = q;				// point at next line
	    }
	    if (*p)
		fprintf(stderr, "< %s\n", p);	// display last line
	}
#endif
	return x;
}
/////////////////////////////////////////////////////////////////////////
char *despace(char *p)
{
	while (*p && (*p <= SPACE))	// find first non-whitespace
	    p++;
	return p;
}
///////////////////////////////////////////////////////////////////////////
int parse_config()
{
	int	i, *x, values_read, count;
	struct	CONFIG *s;
	FILE	*fildes;
	char	*p, *q, *t, *tok;

	for (i=0; i<CONFIGNUM; i++)
	    config[i].seen = 0;			// clear the 'seen' bits
	*groupname = 0;
	*hostname = 0;
	*domain = 0;
	fildes = fopen(config_filename, "r");
	if (fildes == NULL) 
	    return NOCONFIG;
	values_read = 0;
	conf_line = 0;
	while (fgets(buffer, OURBUFLEN, fildes) != NULL) {
	    conf_line++;
	    if ((*buffer == COMMENT) || (*buffer == '\r') || (*buffer == '\n'))
		continue;			// skip comment and blank lines
#ifdef DEBUG
	    if (debug>1)
		fprintf(stderr, "! %s", buffer);
#endif
	    p = strchr(buffer, EQUALS);
	    if (p == NULL)
		return BADCONFIG1;
	    t = p;
	    while ((t>buffer) && (*t <= EQUALS))// find end of keyword
		t--;
	    *(++t) = 0;				// terminate keyword
	    q = despace(p + 1);            	// find start of value
	    p = q;				// point at start of value
	    while (*q && (*q > SPACE) && (*q != ';'))	// find end of dest string
		q++;
	    *q = 0;				// terminate dest string
	    if ((q - p) > (CONFSTRLEN - 2))
		return BADCONFIG2;
	    q = despace(buffer);		// find start of keyword
	    for (i=0; i<CONFIGNUM; i++) {
		s = &config[i];
		if (strcasecmp(s->keyword, q) == 0) {
		    if (s->seen != 0)
			return BADCONFIG3;
		    else
			s->seen = 1;            // we've seen this keyword
#ifdef DEBUG
		    if (debug)
			fprintf(stderr, "! [%s=", s->keyword);
#endif
		    switch (s->type){		// store the value
			case STRING:
			    strcpy(s->dest, p);
#ifdef DEBUG
			    if (debug)
				fprintf(stderr, "%s]\n", (char *)s->dest);
#endif
			    break;
			case BOOL:
			    x = (int *)s->dest;
			    *x = (strncasecmp(p, "Y", 1) == 0);
#ifdef DEBUG
			    if (debug)
				fprintf(stderr, "%d]\n", *((int *)s->dest));
#endif
			    break;
			case NUMBER:
			    x = (int *)s->dest;
			    *x = atoi(p);
			    if (*x <=0 )
				return BADCONFIG6;
#ifdef DEBUG
			    if (debug)
				fprintf(stderr, "%d]\n", *((int *)s->dest));
#endif
			    break;
		    }
		    values_read++;		// one more tag seen
		    break;
		}
	    }
	    if (i == CONFIGNUM)
		return BADCONFIG5;
	}
	if (values_read != CONFIGNUM)
	    return BADCONFIG4;
	return SUCCESS;
}
///////////////////////////////////////////////////////////////////////////
void get_our_visible_IPaddr(char *dest)
{
	int	x;
	char	*p;
	
	*dest = 0;				// remove old address
	if ((x = Connect(CLIENT_IP_PORT)) != SUCCESS)  {
	    if (x != CONNTIMEOUT)		// no message iff offline
		handle_dynup_error(x);
	    return;
	}
	// set up text for web page query
	sprintf(buffer, "GET http://%s/ip.php HTTP/1.0\nUser-Agent: %s DUC %s\n\n", noipname,os,version);
	x = converse_with_web_server();
	if (x < 0) {	
	    handle_dynup_error(x);
	    return;
	}
	close(socket_fd);
	p = strrchr(buffer,'\n');
	if (p) {
	    strcpy(dest, ++p);  	// address start on next line
#ifdef DEBUG
	if (debug)
	    fprintf(stderr,"! Our NAT IP address is %s\n", dest);
#endif
	} else 
	    ErrMsg("Can't get our visible IP address from %s", noipname); 
}
///////////////////////////////////////////////////////////////////////////
int dynamic_update()
{
	int	i, x, fd, retval;
	char	*p;
	char	server_string[256];

	if ((x = Connect(port_to_use)) != SUCCESS) 
	    return x;

	// set up text for web page update
	if (*groupname) {
	    sprintf(server_string, 
		"http://%s/update.php?username=%s&pass=%s&groupname=%s",
		noipname, login, password, ourname);
	} else {
	    sprintf(server_string, 
		"http://%s/update.php?username=%s&pass=%s&host=%s",
		noipname, login, password, ourname);
	}
	sprintf(buffer, "GET %s%s HTTP/1.0\nUser-Agent: %s DUC %s\n\n",server_string, suppliedIP,os,version);
	if ((x = converse_with_web_server()) < 0)	
	    return x;
	close(socket_fd);

	// analyze results
	for (i=0,p=buffer; i<x; i++,p++) {
		// look for updating ip message
		if ((*p == 's') && (!strncmp(p, "status=", 7))) {
			retval = atoi(&p[7]);
			return retval;
		}
	}
	if (x) {		// we got a response with no status line
//		error_filename = tempnam("/tmp", "NO-IP");
//		fd = open(error_filename, O_RDWR | O_TRUNC | O_CREAT, 0644);
		strcpy(error_filename, "/tmp/NO-IPXXXXXX");
		fd = mkstemp(error_filename);
		write(fd, buffer, x);
		close(fd);
	}
	return UNKNOWNERR;
}
/////////////////////////////////////////////////////////////////////////////
int handle_dynup_error(int error_code)
{
	char	*err_string;

	if (error_code == SUCCESS || error_code == SUCCESSGRP) {
	    syslog(LOG_INFO, "%s set to %s", ourname, IPaddress);
	    return SUCCESS;
	}
	err_string = strerror(errno);
	switch (error_code) {
	    case ALREADYSET:
	    case ALREADYSETGRP:
 		syslog(LOG_INFO, "%s was already set to %s.", 
							ourname, IPaddress);
		return SUCCESS;
            case BADHOST:
		ErrMsg("No host '%s' found at %s.", ourname, noipname);
		break;
	    case BADPASSWD:
		ErrMsg(
		"Incorrect password for %s while attempting toupdate %s.", 
							login, ourname);
		break;
	    case BADUSER:
		ErrMsg("No user '%s' found at %s.", login, noipname);
		break;
	    case PRIVATEIP:
		ErrMsg(
			"IP address '%s' is a private network address.", 
								IPaddress);
		ErrMsg("Use the NAT facility.");
		break;	
	    case TOSVIOLATE:
		ErrMsg(
			"Account '%s' has been banned for violating terms of service.",
								login);
		break;
	    case HOSTDISABLED:
		ErrMsg(
                        "Host '%s' has been disabled and cannot be updated.",
                                                                ourname);
		break;
	    case HOSTISREDIRECT:
		 ErrMsg(
                        "Host '%s' not updated because it is a web redirect.",
                                                                ourname);  
                break;							
	    case BADGRP:
                 ErrMsg(
                        "No group '%s' found at %s.",  
                                                                ourname,noipname);
                break;	
	    case RELEASEDISABLED:
                 ErrMsg(
                        "This client software has been disabled.  Please download a new version");
                break;
	    case UNKNOWNERR:
		ErrMsg("Unknown error trying to set %s at %s.", 
							ourname, noipname);
		ErrMsg("Error info saved in %s", error_filename);
		break;
	    case NOHOSTLOOKUP:
		ErrMsg("Can't gethostbyname for %s", noipname);
		break;
	    case SOCKETFAIL:
		ErrMsg("Can't create socket (%s)", err_string);
		break;
	    case CONNTIMEOUT:
		ErrMsg("Connect to %s timed out", noipname);
		break;
	    case CONNFAIL:
		ErrMsg("Can't connect to %s (%s)", noipname, err_string);
		break;
	    case READTIMEOUT:
		ErrMsg("Read timed out talking to %s", noipname);
		break;
	    case READFAIL:
		ErrMsg("Read from %s failed (%s)", noipname, err_string);
		break;
	    case WRITETIMEOUT:
		ErrMsg("Write timed out talking to %s", noipname);
		break;
	    case WRITEFAIL:
		ErrMsg("Write to %s failed (%s)", noipname, err_string); 
		break;
	    default:
		ErrMsg("Unknown error %d trying to set %s at %s.", 
						error_code, ourname, noipname);
		break;
	}
	return -1;
}
///////////////////////////////////////////////////////////////////////////
int handle_config_error(int err_code)
{
	int	i, missing;

	switch(err_code) {
	    case SUCCESS:
		return SUCCESS;
	    case NOCONFIG:
		ErrMsg("Can't locate configuration file %s. (Try -h). Ending!\n", 
							config_filename);
		break;
	    case BADCONFIG1:
		ErrMsg("Missing '=' in %s, line %d. Ending!\n", 
						config_filename, conf_line);
		break;
	    case BADCONFIG2:
		ErrMsg("Value too long in %s, line %d. Ending!\n", 
						config_filename, conf_line);
		break;
	    case BADCONFIG3:
		ErrMsg("Duplicate keyword '%s' in %s, line %d. Ending!\n", 
					buffer, config_filename, conf_line);
		break;
	    case BADCONFIG4:
		for (i=0,missing=0; i<CONFIGNUM; i++) {
		    if (config[i].seen == 0)
			missing++;
		}
		ErrMsg("Missing keyword%s [", (missing == 1) ? "":"s");
		for (i=0; i<CONFIGNUM; i++) {
		    if (config[i].seen == 0)
			ErrMsg( " %s", config[i].keyword); 
		}
		ErrMsg( " ] in %s. Ending!\n", config_filename);
		break;
	    case BADCONFIG5:
		ErrMsg("Unknown keyword '%s' in %s, line %d. Ending!\n", 
					buffer, config_filename, conf_line);
		break;
	    case BADCONFIG6:
		ErrMsg("Negative or zero value in %s, line %d. Ending!\n", 
						config_filename, conf_line);
		break;
	    case BADCONFIG8:
		ErrMsg("Hostname found without domain in %s. Ending!\n", 
							config_filename);
		break;
	    case BADCONFIG9:
		ErrMsg("No hostname or groupname found in %s. Ending!\n", 
							config_filename);
		break;
	}
	return -1;
}
/////////////////////////////////////////////////////////////////////////////
void ErrMsg(char *fmt, ...)
{
	va_list ap;
	char	errmsg[256];

	va_start(ap, fmt);
	vsprintf(errmsg, fmt, ap);
	va_end(ap);
	if (log2syslog)
		syslog(LOG_ERR, "%s\n", errmsg);
	else
		fprintf(stderr, "%s\n", errmsg);
}
/////////////////////////////////////////////////////////////////////////////

